from torch_geometric.datasets import TUDataset
from torch_geometric.data import Data, DataLoader, Batch
from torch_geometric.utils import accuracy

import numpy as np
import torch
import torch.nn.functional as F
from torch.nn import Linear
from torch_geometric.nn import GCNConv, global_mean_pool

device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')

INPUT_DIM   = 1
NUM_CLASSES = 2

class Net(torch.nn.Module):
    def __init__(self):
        super(Net, self).__init__()
        self.conv = GCNConv(INPUT_DIM, 32, normalize=False)
        self.linear = Linear(32, NUM_CLASSES)

    def forward(self, data):
        x, edge_index, batch = data.x, data.edge_index, data.batch

        x = self.conv(x, edge_index)
        x = F.relu(x)
        x = F.dropout(x, training=self.training)

        x = global_mean_pool(x, batch)

        x = self.linear(x)

        return F.log_softmax(x, dim=1)

def generate_tree(n):
    edges = []
    for v in range(1,n) :
        u = np.random.randint( v )
        edges.append( (u,v) )
    perm = np.random.permutation(n)
    edges = [ ( perm[u], perm[v] ) for u,v in edges ]
    edges = [ (u,v) if u<v else (v,u) for u,v in edges ]
    return edges

def generate_connected_graph(n,m):
    edges = set(generate_tree(n))
    while len(edges)<m:
        u,v = np.random.randint(n,size=2)
        if u==v: continue
        if u>v: u,v=v,u
        edges.add((u,v))
    return list(edges)

def to_torch_edge_index(edges):
    edges = [ (u,v) for u,v in edges ] + [ (v,u) for u,v in edges ]
    return torch.tensor(edges, dtype=torch.long).t().contiguous()

dataset = []
for i in range(1000):
    N = np.random.randint( 10,20 )
    dataset.append( Data( x=torch.ones((N,1),dtype=torch.float32), edge_index=to_torch_edge_index(generate_tree(N))             , y=torch.ones (1,dtype=torch.long)) )
for i in range(1000):
    N = np.random.randint( 10,20 )
    dataset.append( Data( x=torch.ones((N,1),dtype=torch.float32), edge_index=to_torch_edge_index(generate_connected_graph(N,N)), y=torch.zeros(1,dtype=torch.long)) )

np.random.shuffle(dataset)
train_data = dataset[:1000]
test_data = dataset[1000:]

model = Net().to(device)
optimizer = torch.optim.Adam(model.parameters(), lr=0.01, weight_decay=5e-4)

def evaluate( model, data ):
    model.eval()
    with torch.no_grad():
        batch = Batch.from_data_list( data ).to(device)
        out = model( batch )
        _, pred = out.max(dim=1)
        acc  = accuracy( pred, batch.y )
        loss = F.nll_loss(out, batch.y).item()
    model.train()
    return acc,loss

model.train()
for epoch in range(500):
    if epoch % 100 == 0:
        print( 'EPOCH', epoch )
        print( 'train acc,loss:', *evaluate(model,train_data))
        print( 'test  acc,loss:', *evaluate(model,test_data ))

    loader = DataLoader(train_data, batch_size=32, shuffle=True)
    for batch in loader:
        optimizer.zero_grad()
        batch = batch.to(device)
        out = model(batch)
        loss = F.nll_loss(out, batch.y)
        loss.backward()
        optimizer.step() 

print( 'END' )
print( 'train acc,loss:', *evaluate(model,train_data))
print( 'test  acc,loss:', *evaluate(model,test_data ))


"""
EPOCH 0
train acc,loss: 0.484 0.7262688875198364
test  acc,loss: 0.516 0.7127169370651245
EPOCH 100
train acc,loss: 0.997 0.3655383586883545
test  acc,loss: 0.994 0.3677332103252411
EPOCH 200
train acc,loss: 0.997 0.3368944525718689
test  acc,loss: 0.996 0.341035932302475
EPOCH 300
train acc,loss: 0.999 0.32543426752090454
test  acc,loss: 0.998 0.32746708393096924
EPOCH 400
train acc,loss: 0.991 0.341607928276062
test  acc,loss: 0.99 0.3365638256072998
END
train acc,loss: 0.996 0.3243809640407562
test  acc,loss: 0.992 0.33117780089378357
"""
