from torch_geometric.datasets import TUDataset
from torch_geometric.data import DataLoader, Batch
from torch_geometric.utils import accuracy

import torch
import torch.nn.functional as F
from torch.nn import Linear
from torch_geometric.nn import GCNConv, global_mean_pool

device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')

class Net(torch.nn.Module):
    def __init__(self):
        super(Net, self).__init__()
        self.conv = GCNConv(dataset.num_node_features, 32)
        self.linear = Linear(32, dataset.num_classes)

    def forward(self, data):
        x, edge_index, batch = data.x, data.edge_index, data.batch

        x = self.conv(x, edge_index)
        x = F.relu(x)
        x = F.dropout(x, training=self.training)

        x = global_mean_pool(x, batch)

        x = self.linear(x)

        return F.log_softmax(x, dim=1)

dataset = TUDataset(root='.', name='PROTEINS')
dataset = dataset.shuffle()
train_data = dataset[:800]
test_data = dataset[800:]

model = Net().to(device)
optimizer = torch.optim.Adam(model.parameters(), lr=0.01, weight_decay=5e-4)

def evaluate( model, data ):
    model.eval()
    with torch.no_grad():
        batch = Batch.from_data_list( data ).to(device)
        out = model( batch )
        _, pred = out.max(dim=1)
        acc  = accuracy( pred, batch.y )
        loss = F.nll_loss(out, batch.y).item()
    model.train()
    return acc,loss

model.train()
for epoch in range(500):
    if epoch % 100 == 0:
        print( 'EPOCH', epoch )
        print( 'train acc,loss:', *evaluate(model,train_data))
        print( 'test  acc,loss:', *evaluate(model,test_data ))

    loader = DataLoader(train_data, batch_size=32, shuffle=True)
    for batch in loader:
        optimizer.zero_grad()
        batch = batch.to(device)
        out = model(batch)
        loss = F.nll_loss(out, batch.y)
        loss.backward()
        optimizer.step() 

print( 'END' )
print( 'train acc,loss:', *evaluate(model,train_data))
print( 'test  acc,loss:', *evaluate(model,test_data ))

"""
EPOCH 0
train acc,loss: 0.5975 0.681484580039978
test  acc,loss: 0.5910543130990416 0.6826412677764893
EPOCH 100
train acc,loss: 0.685 0.6251936554908752
test  acc,loss: 0.7060702875399361 0.6068294048309326
EPOCH 200
train acc,loss: 0.6825 0.6258826851844788
test  acc,loss: 0.7380191693290735 0.6042547225952148
EPOCH 300
train acc,loss: 0.695 0.6270562410354614
test  acc,loss: 0.744408945686901 0.6041122078895569
EPOCH 400
train acc,loss: 0.6875 0.6252518892288208
test  acc,loss: 0.7220447284345048 0.6061394810676575
END
train acc,loss: 0.68 0.6253974437713623
test  acc,loss: 0.7156549520766773 0.6054355502128601
"""
